package org.masstheory;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.Set;
import java.util.Vector;

public class Pioneer {

  
  DlgMain dlg = null;
  
  double k = 6.67428e-11;     // gravity constant
  double M = 1.98892e30;      // mass of the Sun
  double AU = 1.49598e11;     // astronomical unit
  double c = 2.998e8;         // speed of light
  
  // Calculate dynamic acceleration of gravity for pioneers 
  // between these distances from the Sun
  int startDistanceAU = 10, endDistanceAU = 70;
  
  public Pioneer(DlgMain dlg) {
    this.dlg = dlg;
    loadAndProcessCoordinates("pioneer_10.txt");
  }
  
  /** 
   * Calculates dynamic gravity acceleration using 
   * Pioneer spacecraft speed and coordinates 
   */
  void loadAndProcessCoordinates(String fileName) {
    
    // times (as milliseconds) and distances from Sun (as AU) for each coordinate
    LinkedHashMap<Long, Double> map1 = new LinkedHashMap<Long, Double>();
    
    // times (as milliseconds) and speeds (as m/s)
    LinkedHashMap<Long, Double> map2 = new LinkedHashMap<Long, Double>();

    // times (as milliseconds) and cos phi
    LinkedHashMap<Long, Double> map3 = new LinkedHashMap<Long, Double>();
    
    try {
      String fileContents = loadResourceFile(fileName);
      if (fileContents.isEmpty()) {
        System.out.println("Coordinate file " + fileName + " not available as resource.");
        fileContents = loadTextFile(fileName);
      }
      
      if (fileContents.isEmpty()) {
        System.out.println("Coordinate file " + fileName + " not found.");
        System.exit(1);
      }
      
      BufferedReader in = new BufferedReader(new StringReader(fileContents));
      Vector<String> lines = new Vector<String>();
      
      boolean afterStartMarker = false;
      String line = null;
      while ( (line = in.readLine()) != null ) {
        if (!afterStartMarker) {
          if (line.equals("$$SOE")) {
            afterStartMarker = true;
          }
          
          continue;
        }
        
        if (line.equals("$$EOE")) {
          break;
        }
        
        lines.add(line);
      }
      
      // this strictly depends on the format of 
      // the coordinate files which come with this software
      for (int i = 0; i < lines.size(); i++) {
        line = lines.elementAt(i);
        
        if (!line.contains("A.D.")) {
          continue;
        }

        // parsing line like this:
        // 2441803.500000000 = A.D. 1973-May-01 00:00:00.0000 (CT)
        String date = line.substring(line.indexOf("A.D.") + 4).trim();
        date = date.substring(0, date.indexOf(" "));
        
        String[] fields = date.split("-");
        
        int year = Integer.parseInt(fields[0]);
        int month = getMonth(fields[1]);
        int day = Integer.parseInt(fields[2]);
        
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.MONTH, month);
        cal.set(Calendar.DAY_OF_MONTH, day);
        
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.MILLISECOND, 0);        
        
        // next line holds X Y Z relative to the center of the Sun
        // line like this:
        // -7.259513200571780E-01 -7.790015164896215E-01 -2.902600882619302E-02
        line = lines.elementAt(i + 1);
        String[] xyz = line.trim().split("\\s+");
        
        Double x = Double.parseDouble(xyz[0]);
        Double y = Double.parseDouble(xyz[1]);
        Double z = Double.parseDouble(xyz[2]);
        
        double distanceFromSun = Math.sqrt(Math.pow(x,2) + Math.pow(y,2) + Math.pow(z,2));
        
        if (distanceFromSun < startDistanceAU || distanceFromSun > endDistanceAU) {
          continue;
        }
        
        // next line holds velocity components VX VY VZ
        // line like this:
        // 1.215273683571856E-02 -1.792764141673417E-02 -1.103442075216277E-03
        line = lines.elementAt(i + 2);
        String[] vxvyvz = line.trim().split("\\s+");
        
        Double vx = Double.parseDouble(vxvyvz[0]);
        Double vy = Double.parseDouble(vxvyvz[1]);
        Double vz = Double.parseDouble(vxvyvz[2]);
        
        // this will be AU/day
        double speedAUperDay = Math.sqrt(Math.pow(vx,2) + Math.pow(vy,2) + Math.pow(vz,2));
        
        // and this will be m/s
        double speed = speedAUperDay * AU / 86400;
        
        
        // Now we can calculate cos phi where phi is angle 
        // between position vector and velocity
        double cosphi = (x * vx + x * vy + z  * vz) / distanceFromSun / speedAUperDay;
        
        Long millis = cal.getTimeInMillis();
        
        map1.put(millis, distanceFromSun);
        map2.put(millis, speed);
        map3.put(millis, cosphi);
        
        //double degrees = round(Math.acos(cosphi) / Math.PI * 180d, 1);
        //System.out.println(distanceFromSun + "\t\tspeed: " + speed + "\t\tdegrees:" + degrees);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    
    Set<Long> keys = map1.keySet();
    Long[] array = keys.toArray(new Long[0]);
    
    // total distance travelled towards the Sun due to dynamic gravity acceleration
    double totalDrift = 0;
    // total speed acquired by spacecraft due to this acceleration
    double totalDriftSpeed = 0; 
    // time between start and end positions of the spacecraft
    double totalTimeSec = 0;
    
    // period between coordinates in this array
    long periodSec = (array[1] - array[0]) / 1000;
    
    StringBuilder b = new StringBuilder();
    for (int i = 0; i < array.length; i++) {
      long millis = array[i];
      
      double distanceFromSunAU = map1.get(array[i]);
      double distanceFromSun = distanceFromSunAU * AU; // distance in m/s
      
      double speed = map2.get(millis);
      double cosphi = map3.get(millis);
      
      // dynamic gravity acceleration as explained in "OSNOVNE OSOBINE SVETA 3"
      //double acceleration = k * M / Math.pow(distanceFromSun, 2) * 2 * speed / c * cosphi; // this is simplified formula
      double acceleration = k * M / Math.pow(distanceFromSun, 2) * 
        (- Math.pow(speed, 2) + 2 * speed * c * cosphi) / 
        (Math.pow(c, 2) + Math.pow(speed, 2) - 2 * speed * c * cosphi);
      
      // distance travelled towards the Sun due to dynamic acceleration
      double additionToDriftSpeed = acceleration * periodSec;
      totalDriftSpeed += additionToDriftSpeed;
      
      // distance travelled towards the Sun due to this acceleration
      double drift = totalDriftSpeed * periodSec + acceleration * Math.pow(periodSec, 2) / 2;
      totalDrift += drift;
      totalTimeSec += periodSec;

      // aling right for nicer output
      String roundAcceleration = "" + round(acceleration/1e-10, 2);
      while (roundAcceleration.length() < 10) {
        roundAcceleration = " " + roundAcceleration;
      }

      b.append("  ").append(round(distanceFromSunAU, 1)).append("\t").append(roundAcceleration).append("\n");
      
      //System.out.println(round(distanceFromSunAU, 1) + "\t\t" + round(speed, 1) + "\t\t" + roundAcceleration);
    }
       
    dlg.textArea.setText(b.toString());
    dlg.scrollableText.getVerticalScrollBar().setValue(dlg.scrollableText.getVerticalScrollBar().getMaximum());
    
    double a = 2 * totalDrift / Math.pow(totalTimeSec, 2);
    
    dlg.textDrift.setText("" + round(totalDrift/1000, 1));
    dlg.textDriftSpeed.setText("" + round(totalDriftSpeed, 2));
    dlg.textTime.setText("" + round(totalTimeSec/86400/365, 1));
    dlg.textEquivAcc.setText("" + round(a/1e-10,2));
    
    //System.out.println();
    //System.out.println("Total drift: " + round(totalDrift/1000, 1) + " Km");
    //System.out.println("Total time: " + round(totalTimeSec/86400/365, 1) + " years");
    //System.out.println("Total drift speed: " + round(totalDriftSpeed, 2) + " m/s");
    //System.out.println("Equivalent constant acceleration: " + round(a/1e-10,2) + "E-10 m/s^2");
  }
  
  int getMonth(String month) {
    
    String[] months = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    
    for (int i = 0; i < months.length; i++) {
      if (months[i].equals(month)) {
        return i;
      }
    }
    
    return -1;
  }
  
  /** rounds a number to specified number of decimal places */
  double round(double value, int decimals) {
    
    return Math.round(value * Math.pow(10, decimals)) / Math.pow(10, decimals);
  }
  
  String loadResourceFile(String fileName) {
    
    if (!fileName.startsWith("/")) {
      fileName = "/" + fileName;
    }
    
    StringBuilder b = new StringBuilder();
    
    try {
      InputStream is = getClass().getResourceAsStream(fileName);
      
      if (is != null) {
        for(int c = is.read(); c != -1; c = is.read()) {
          b.append((char)c);
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    
    return b.toString();
  }
  
  String loadTextFile(String fileName) {
    
    StringBuilder b = new StringBuilder();
    
    try {
      BufferedReader in = new BufferedReader(new FileReader(fileName));
      
      String line = null;
      while ( (line = in.readLine()) != null ) {
        b.append(line).append("\n");
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    
    return b.toString();
  }
}